; Control.asm - demonstrates child window controls
;
; Child window controls are an extension to child windows. As outlined before,
; child windows can be used as input devices - they handle the actual input and
; then notify the parent window by messages, which does the actual work (perhaps
; sending messages back to the child windows to change their display). So, the
; question now is how do we make a child window into a control?
;
; First, we'd better define what a control is - it's a generic name for a set
; of GUI widgets that comprise:
;
;   Buttons - a wide variety of buttons, covering a range of situations:
;     Push buttons
;     Checkboxes
;     Radio buttons
;     Group boxes
;   Comboboxes - Like listboxes but with an edit control attached
;   Edit controls - both single and multiline
;   Listboxes
;   Scrollbars
;   Static controls - Things like text and graphics
;
; The group boxes are another example of Microsoft's strange ideas - they're not
; a button at all, but are used to group controls together. Still, in the
; buttons is where they're found.
;
; The above are the core GUI widgets, the core controls available to Windows
; applications, but as Windows developed it became clear that they weren't going
; to be enough and developers and designers wanted new controls to make a better
; GUI and also to keep up with other GUI developments. These extra widgets (which
; are generally more advanced than the original controls) were formed into a new
; set of controls called, slightly oddly, 'common controls', and consist of things
; like the TreeView control, toolbars, progress bars, etc.
;
; In this section, I'll demonstrate the core controls only, and save the common
; controls until later. In this first example, I'll demonstrate a simple button,
; and a static control so you can get an idea of how controls work. Also, I'll
; demonstrate sending messages to the child window from the parent window.
;
; So, how do we actually create a child window control?
;
; Recall how when we created a child window before, we registered a window class
; for each child window (well, one class for both of them but the number isn't
; important) - all that matters is that a child window uses a window class.
;
; Child window controls are just the same, except that there are _predefined_
; window classes for each control. Hence, you never need to register a window
; class for them and when you call CreateWindowEx you pass the class name of
; the control you want to use. These are all predefined by Windows, but to save
; you the trouble they've been defined in Windows.inc, and can be referred to by:
;
;   _szClassNameButton
;   _szClassNameCombobox
;   _szClassNameEdit
;   _szClassNameListbox
;   _szClassNameScrollbar
;   _szClassNameStatic
;
; Just pass these to CreateWindowEx.
;
; Now all that's left is how to respond to messages sent by the control.
; Normally, we'd have our own window procedure for the child but in this case,
; Windows takes care of that. Instead, it sends a message directly to the
; parent, and the parent deals with it from there. This will all be explained
; in the window procedure below.
;
; If you examine the resulting application, you'll notice that the controls
; are colored as if they were on a dialog box. Also, you'll notice that once
; a control has the focus (the button) you can't tab to another control (not
; that it matters in this example as it's just a static control which can't
; receive the focus anyway but it's the principle that's important).
;
; These problems arise because controls were originally designed for dialog
; boxes only, and it can be hard work making them behave like on a dialog box.
; We can solve these issues by making a dialog box _the main window_, but
; we'll leave that until later. For now, concentrate on understanding child
; window controls as these are the building blocks of all dialog boxes (and
; the common controls merely add flavour to these).
;

%define _WINMESSAGES_
%include "Gaz\Win32\Include\Windows.inc"

[BITS 32]
[section .text]

ddglobal _gbl_hInstance

procglobal WinMain, hInstance, hPrevInstance, lpszCmdLine, nCmdShow
	ddlocal		_hwnd
	struclocal	_wndclass, WNDCLASSEX, _msg, MSG
	endlocals
	WinMainPrologue
	mov	eax, .hInstance
	mov	._gbl_hInstance, eax
	mov	esi, ._wndclass
	mov	edi, ._msg
	mov	[esi + WNDCLASSEX.cbSize], dword WNDCLASSEX_size
	mov	[esi + WNDCLASSEX.style], dword CS_HREDRAW | CS_VREDRAW
	mov	[esi + WNDCLASSEX.lpfnWndProc], dword _WndProc
	mov	[esi + WNDCLASSEX.cbClsExtra], dword 0
	mov	[esi + WNDCLASSEX.cbWndExtra], dword 0
	mov	eax, .hInstance
	mov	[esi + WNDCLASSEX.hInstance], eax
	sc LoadIcon, NULL, IDI_APPLICATION
	mov	[esi + WNDCLASSEX.hIcon], eax
	sc LoadCursor, NULL, IDC_ARROW
	mov	[esi + WNDCLASSEX.hCursor], eax
	sc GetStockObject, WHITE_BRUSH
	mov	[esi + WNDCLASSEX.hbrBackground], eax
	mov	[esi + WNDCLASSEX.lpszMenuName], dword NULL
	TEXTlocal szClassName, 'MyClass',0
	mov	[esi + WNDCLASSEX.lpszClassName], dword .szClassName
	sc RegisterClassEx, esi
	cmp	eax, TRUE
	je	near _WinMain_Fail
	TEXTlocal szWndCaption, 'Child window controls',0
	sc CreateWindowEx, 0, .szClassName, .szWndCaption, WS_OVERLAPPEDWINDOW | WS_VISIBLE, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, NULL, NULL, .hInstance, NULL
	mov	._hwnd, eax
	sc ShowWindow, ._hwnd, .nCmdShow
	sc UpdateWindow, ._hwnd
_WinMain_Loop:
	sc GetMessage, ._msg, NULL, 0, 0
	cmp	eax, TRUE
	jne	_WinMain_Loop_End
	sc TranslateMessage, ._msg
	sc DispatchMessage, ._msg
	jmp	_WinMain_Loop
_WinMain_Loop_End:
	mov	eax, [edi + MSG.wParam]
	jmp	_WinMain_End
_WinMain_Fail:
	TEXTlocal szErrorMsg, 'Failed to register window class!',0
	sc MessageBox, NULL, .szErrorMsg, .szWndCaption, MB_ICONERROR
_WinMain_End:
	WinMainEpilogue
endproc
;
;-----------------------------------------------------------------------
;
proc _WndProc, hwnd, message, wParam, lParam
	ddstatic	_hwnd1, _hwnd2
	endlocals
	;
	CallbackPrologue
	;
	switch .message
		case WM_CREATE
			;
			; Create the first child control. Note that there are
			; very few changes required to turn this into a control -
			; the second parameter now points to the class name for the
			; control (the button control in this case) and the third
			; parameter is used to specify the text for the button rather
			; than the window title. For simplicity, I've set this to the
			; same as the class name. Also note that I've added a new
			; paramter to the style - the type of button I want to display.
			; For each control there is usually a range of styles to choose
			; from, all listed in the API reference.
			;
			sc CreateWindowEx, 0, _szClassNameButton, _szClassNameButton, WS_CHILDWINDOW | WS_VISIBLE | BS_PUSHBUTTON, 0, 0, 0, 0, .hwnd, 1, ._gbl_hInstance, NULL
			;
			; Having created the child window, we store it's window handle
			;
			mov	._hwnd1, eax
			;
			; And now create the second child control. In this one, I've
			; used the static control. Again, I've set the text to the
			; class name for simplicity and added a static style. Consult
			; the API reference for the full list of control styles. You'll
			; find them under the CreateWindow call.
			;
			sc CreateWindowEx, 0, _szClassNameStatic, _szClassNameStatic, WS_CHILDWINDOW | WS_VISIBLE | SS_CENTER, 0, 0, 0, 0, .hwnd, 2, ._gbl_hInstance, NULL
			;
			; And store its window handle
			;
			mov	._hwnd2, eax
			xor	eax, eax
			break
		case WM_COMMAND
			;
			; Message from a child window control?
			;
			; These are always sent by WM_COMMAND messages. The message
			; parameters itself contain the details of which child window
			; control sent it and what the message is:
			;
			;   wParam (low word) is the child window's ID
			;   wParam (high word) is the notification code
			;   lParam is the child window's handle
			;
			; The first and last are already known to us, but the second isn't.
			; Just what is this notification code? Essentially, it's a mini-
			; message informing us what action has just been taken. If you look
			; in the API reference under buttons, you can find a Button messages
			; topic which lists all the messages a button can send and receive.
			; If you look closely, you'll see some are prefixed BN, which stands
			; for Button Notification, and it's these that we're interested in.
			; Every control has notification codes, and again you can find all
			; the details in the API reference.
			;
			; Anyway, first thing to do is work out which window this is. In
			; actual fact though, in this simple example there's no need to
			; even bother - the static control never sends notification messages
			; unless we explicitly ask it to (add SS_NOTIFY style when you create
			; the control if you want this)
			;
			; Knowing the notification must be from the button, we should get the
			; actual notification code in eax:
			;
			mov	ebx, .wParam
			shr	ebx, 16
			movsx	eax, bx
			;
			; Was the button clicked? (The only notification we're interested in)
			;
			if	eax, e, BN_CLICKED
				;
				; Yes, so for this example, I'm going to make the static control change
				; text to a different string. To accomplish this, I call the
				; SetWindowText function which takes two parameters:
				;
				;   1. Handle to the window to change
				;   2. Pointer to the new text string
				;
				; It doesn't matter that the window is a child window, or that it's a
				; child window control. If it's a control, the control's text is
				; changed rather than the window title. We just need to ensure we pass
				; the child window control's window handle in the function. For
				; simplicity, I've just set the text to being the button class name:
				;
				sc SetWindowText, ._hwnd2, _szClassNameButton
				;
				; Having done this, I now demonstrate sending a message to a child
				; window control. In this case, I'm going to change the button style
				; to being a checkbox. A silly example, but it serves the purpose.
				;
				; We send a message using the SendMessage function which requires
				; four parameters:
				;
				;   1. Handle of the window to send the message to
				;   2. Message to send
				;   3. wParam
				;   4. lParam
				;
				; I'm going to send BM_SETSTYLE as the message with BS_CHECKBOX for
				; wParam and TRUE for lParam, which tells Windows to redraw the
				; button. Like with everything else, you can find the full details
				; of this message in the API reference.
				; 
				sc SendMessage, ._hwnd1, BM_SETSTYLE, BS_AUTOCHECKBOX, TRUE
			endif
			xor	eax, eax
			break
		case WM_SIZE
			mov	ebx, .lParam
			movsx	esi, bx
			shr	ebx, 16
			movsx	edi, bx
			shr	esi, 1
			sc MoveWindow, ._hwnd1, 0, 0, esi, edi, TRUE
			sc MoveWindow, ._hwnd2, esi, 0, esi, edi, TRUE
			xor	eax, eax
			break
		case WM_DESTROY
			sc PostQuitMessage, 0
			xor	eax, eax
			break
		default
			sc DefWindowProc, .hwnd, .message, .wParam, .lParam
	switchend
	;
	CallbackEpilogue
endproc
;
;-----------------------------------------------------------------------
;
[section .bss]

[section .data]
